package net.minecraft.mypeople;

import java.util.ArrayList;
import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityEggInfo;
import net.minecraft.entity.EntityList;
import net.minecraft.entity.EntityLiving;
import net.minecraft.item.ItemStack;
import net.minecraft.mypeople.entity.EntityCustomized;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.Facing;
import net.minecraft.util.StatCollector;
import net.minecraft.world.World;

public class SpawnData {

	
	public boolean loadedSuccesfully = false;
	
	public SpawnData(){
		
	}
	
	public SpawnData(NBTTagCompound nbt){
		loadFrom(nbt);
	}
	
	public void saveTo(NBTTagCompound nbt){
		nbt.setIntArray("mixedEntityIds", mixedEntityIds);
		nbt.setInteger("colour1", colour1);
		nbt.setInteger("colour2", colour2);
		nbt.setInteger("colour3", colour3);
		nbt.setBoolean("incubated", incubated);
		NBTTagCompound vars = new NBTTagCompound();
		writeVars(vars);
		nbt.setTag("Variables", vars);
	}
	
	public void writeVars(NBTTagCompound nbt){
		nbt.setIntArray("mixedIds", mixedEntityIds);
		nbt.setInteger("attackStrength", attackStrength);
		nbt.setInteger("maxHealth", maxHealth);
		nbt.setInteger("healRate", healRate);
		nbt.setFloat("size", size);
		nbt.setFloat("topMoveSpeed", topMoveSpeed);
		nbt.setFloat("jumpStrength", jumpMultiplier);
		nbt.setBoolean("breatheUnderwater", breatheUnderwater);
		nbt.setBoolean("catchOnFire", catchOnFire);
	}
	
	public void loadFrom(NBTTagCompound nbt){
		loadedSuccesfully = true;
		try{
			filter();
			colour1 = nbt.getInteger("colour1");
			colour2 = nbt.getInteger("colour2");
			colour3 = nbt.getInteger("colour3");
			incubated = nbt.getBoolean("incubated");
			NBTTagCompound vars = (NBTTagCompound)nbt.getTag("Variables");
			if(vars != null){
				readVars(vars);
			}
		}catch(Exception e){
			loadedSuccesfully = false;
		}
	}

	public void readVars(NBTTagCompound nbt){
		mixedEntityIds = nbt.getIntArray("mixedIds");
		attackStrength = nbt.getInteger("attackStrength");
		maxHealth = nbt.getInteger("maxHealth");
		healRate = nbt.getInteger("healRate");
		size = nbt.getFloat("size");
		topMoveSpeed = nbt.getFloat("topMoveSpeed");
		jumpMultiplier = nbt.getFloat("jumpStrength");
		breatheUnderwater = nbt.getBoolean("breatheUnderwater");
		catchOnFire = nbt.getBoolean("catchOnFire");
	}
	
	public void spawnEntity(World world, int x, int y, int z, int side){
        int i1 = world.getBlockId(x, y, z);
        if(side > -1){
            x += Facing.offsetsXForSide[side];
            y += Facing.offsetsYForSide[side];
            z += Facing.offsetsZForSide[side];
        }

        double d0 = 0.0D;

        if (side == 1 && Block.blocksList[i1] != null && Block.blocksList[i1].getRenderType() == 11)
        {
            d0 = 0.5D;
        }
        spawnAt(world, x+.5, y+d0, z+.5);
	}
	
	public void spawnEntity(World world, double x, double y, double z){
		spawnAt(world, x, y, z);
	}

	
	public void spawnEntity(World world, double x, double y, double z, int side){
		if(side == 1){
			y--;
		}
		spawnEntity(world, (int)Math.floor(x), (int)Math.floor(y), (int)Math.floor(z), side);
	}
	
	private void spawnAt(World world, double x, double y, double z){
		if(world.isRemote){
			return;
		}
		Entity e = createEntity(world);
		if(e == null){
			System.out.println("Entity is null when spawning. Did a name change?");
			return;
		}
		e.setPosition(x, y, z);
//		System.out.println("Spawning " + e + " in world at " + x + ", " + y + ", " + z);
		world.spawnEntityInWorld(e);
	}
	
	public Entity createEntity(World world){
		if(mixedEntityIds.length == 1){
			return EntityList.createEntityByID(mixedEntityIds[0], world);
		}else if(mixedEntityIds.length > 1){
			return new EntityCustomized(world).load(this);
		}
		return null;
	}

	public NBTTagCompound toTagCompound() {
		NBTTagCompound nbt = new NBTTagCompound();
		this.saveTo(nbt);
		return nbt;
	}
	
	public String getMixedEntityName(){
		String ret = "";
		if(mixedEntityIds.length == 1){
			ret = getName(mixedEntityIds[0]);
		}else{
			for(int a = 0; a < mixedEntityIds.length; a++){
				String name = getName(mixedEntityIds[a]);
				int length = (int)Math.round(name.length()/2);
				int left = name.length() - length;
				int start = (int)Math.round(left * (((float)a)/((float)mixedEntityIds.length)));
				int end = start+length;
				if(a+1 == mixedEntityIds.length){
					end = name.length();
				}
				String append = name.substring(start, end);
				
				ret += append.toLowerCase();
			}
		}
		if(ret != null && ret.length() > 0){
			ret = ret.substring(0, 1).toUpperCase() + ret.substring(1);
		}
		return ret;
	}
	
	public String getName(int entityID){
		if(entityID == 0){
			return "Clone";
		}
		String nameOrig = EntityList.getStringFromID(entityID);
		String name = nameOrig;
		if(name != null){
			name = StatCollector.translateToLocal("entity." + name + ".name");
			if(name == null){
				name = nameOrig;
			}
		}
		if(name == null){
			name = "";
		}
		return name;
	}

	public String getDisplayNameCapsule() {
/*		String ret = (incubated?"":"Unincubated ");
		ret += getMixedEntityName();
		ret += " Capsule";*/
		String ret = getMixedEntityName();
		ret += (incubated?" Capsule":" DNA");
		return ret;
	}
	
	public String[] getData(){
		ArrayList<String> lines = new ArrayList<String>();
		if(mixedEntityIds.length > 1){
			String entities = "";
			for(int a = 0; a < mixedEntityIds.length; a++){
				boolean hasColour = false;
				char colour = '7';
				if(EntityList.entityEggs.containsKey(mixedEntityIds[a])){
					EntityEggInfo info = (EntityEggInfo)EntityList.entityEggs.get(mixedEntityIds[a]);
					if(info != null){
						hasColour = true;
						colour = getClosestColour(info.primaryColor);
					}
				}
				if(hasColour){
					entities += "\247" + colour + EntityList.getStringFromID(mixedEntityIds[a]) + "\2477";
				}else{
					entities += getName(mixedEntityIds[a]);
				}
				
				
				if(a+1 != mixedEntityIds.length){
					entities += ", ";
				}
			}
			lines.add(entities);
		}
		
		return lines.toArray(new String[lines.size()]);
	}
	
	private char getClosestColour(int colour){
		int[] colours = new int[]{0x000000, 0x0000aa, 0x00aa00, 0x00aaaa, 0xaa0000, 0xaa00aa, 0xffaa00, 0x2a2a2a, 0x555555, 0x5555ff, 0x55ff55, 0x55ffff, 0xff5555, 0xff55ff, 0xffff55, 0xffffff};
		char[] colourCodes = "0123456789abcdef".toCharArray();
		int r1 = (colour >> 16) & 0xff;
		int g1 = (colour >> 8) & 0xff;
		int b1 = colour & 0xff;
		
		int closestIndex = -1;
		int diff = -1;
		for(int a = 0; a < colours.length; a++){
			int r2 = (colours[a] >> 16) & 0xff;
			int g2 = (colours[a] >> 8) & 0xff;
			int b2 = colours[a] & 0xff;
			
			int diffR = r1>r2?r1-r2:r2-r1;
			int diffG = g1>g2?g1-g2:g2-g1;
			int diffB = b1>b2?b1-b2:b2-b1;
			
			int totalDiff = diffR + diffG + diffB;
			if(closestIndex < 0 || totalDiff < diff){
				diff = totalDiff;
				closestIndex = a;
			}
		}
		if(closestIndex > -1){
			return colourCodes[closestIndex];
		}
		return ' ';
	}
	

	public void updateColours(){
		if(mixedEntityIds.length == 1 && EntityList.entityEggs.containsKey(mixedEntityIds[0])){
			EntityEggInfo info = (EntityEggInfo)EntityList.entityEggs.get(mixedEntityIds[0]);
			colour1 = info.primaryColor;
			colour2 = info.secondaryColor;
			colour3 = 0x666666;
		}else{
			ArrayList<Integer> colourBin = new ArrayList<Integer>();
			for(int a = 0; a < mixedEntityIds.length; a++){
				EntityEggInfo info = (EntityEggInfo)EntityList.entityEggs.get(mixedEntityIds[a]);
				if(info != null){
					colourBin.add(info.primaryColor);
				}
			}
			if(colourBin.size() < 3){
				for(int a = 0; a < mixedEntityIds.length; a++){
					EntityEggInfo info = (EntityEggInfo)EntityList.entityEggs.get(mixedEntityIds[a]);
					if(info != null){
						colourBin.add(info.secondaryColor);
					}
				}
			}
			if(colourBin.size() < 3){
				for(int a = colourBin.size(); a < 3; a++){
					colourBin.add(0xffffff);
				}
			}
			int total = 0;
			for(int a = 0; a < mixedEntityIds.length; a++){
				total += mixedEntityIds[a];
			}
			Random random = new Random(total);

			colour1 = colourBin.remove(random.nextInt(colourBin.size()));
			colour2 = colourBin.remove(random.nextInt(colourBin.size()));
			colour3 = colourBin.remove(random.nextInt(colourBin.size()));
		}

	}
	
	
	
	//Data
	boolean incubated = false;
	private int colour1 = 0xffffff, colour2 = 0xffffff, colour3 = 0xffffff;
	
	
	public int[] mixedEntityIds = new int[]{};

	public int attackStrength, maxHealth = 20, healRate = 1;
	public float size = 1.0f, topMoveSpeed = .32f, jumpMultiplier = 1.0f;
	public boolean breatheUnderwater = false, catchOnFire = false;
	

	
	public SpawnData setMixedEntityIds(int... ids){
		for(int a = 0; a < ids.length; a++){
			if(ids[a] == 0){
				ids[a] = MyPeople.cloneID;
			}
		}
		mixedEntityIds = ids;
		filter();
		updateColours();
		return this;
	}
	
	private void filter(){
		ArrayList<Integer> dnaIds = new ArrayList<Integer>();
		for(int a = 0; a < mixedEntityIds.length; a++){
			if(!dnaIds.contains(mixedEntityIds[a])){
				dnaIds.add(mixedEntityIds[a]);
			}
		}
		int[] array = new int[dnaIds.size()];
		for(int a = 0; a < dnaIds.size(); a++){
			array[a] = dnaIds.get(a);
		}
		mixedEntityIds = array.clone();
	}
	
	public SpawnData setMixedEntityIds(ArrayList<Integer> list){
		int[] ids = new int[list.size()];
		for(int a = 0; a < list.size(); a++){
			ids[a] = list.get(a);
		}
		for(int a = 0; a < ids.length; a++){
			if(ids[a] == 0){
				ids[a] = MyPeople.cloneID;
			}
		}
		mixedEntityIds = ids;
		filter();
		updateColours();
		return this;
	}

	public int[] getMixedEntityIds(){
		return mixedEntityIds;
	}
	
	public static SpawnData get(ItemStack from){
		if(from != null){
			return new SpawnData(from.stackTagCompound);
		}
		return null;
	}

	public int getColour(int pass) {
		switch(pass){
		case 0: return colour3;
		case 1: return colour2;
		case 2: return colour1;
		}
		return new Random().nextInt();
	}
	
	public boolean canPlace(){
		return incubated;
	}
	
	public SpawnData setPlaceable(boolean flag){
		incubated = flag;
		return this;
	}

	public static boolean isUnincubated(ItemStack stack) {
		if(stack != null){
			SpawnData data = SpawnData.get(stack);
			if(data.loadedSuccesfully){
				return !data.incubated;
			}
		}
		return false;
	}
	
}
