/**
 * Copyright 2014 Yamato
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package mod.ymt.air.cmn.fml;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.MessageToMessageCodec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import net.minecraft.client.Minecraft;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.network.NetHandlerPlayServer;
import cpw.mods.fml.common.FMLCommonHandler;
import cpw.mods.fml.common.network.FMLEmbeddedChannel;
import cpw.mods.fml.common.network.FMLOutboundHandler;
import cpw.mods.fml.common.network.NetworkRegistry;
import cpw.mods.fml.common.network.internal.FMLProxyPacket;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

// refer to: http://www.minecraftforge.net/wiki/Netty_Packet_Handling
// refer to: http://forum.minecraftuser.jp/viewtopic.php?f=21&t=18255

@ChannelHandler.Sharable
public abstract class PacketPipeline extends MessageToMessageCodec<FMLProxyPacket, AbstractPacket> {
	private EnumMap<Side, FMLEmbeddedChannel> channels;
	private List<Class<? extends AbstractPacket>> packets = new ArrayList<Class<? extends AbstractPacket>>();
	private boolean initializing = false;
	
	public boolean registerPacket(Class<? extends AbstractPacket> clazz) {
		if (initializing == false) {
			throw new IllegalStateException("PacketPipeline is not initializing");
		}
		if (this.packets.size() > 256) {
			// パケットの種類を byte 値で送信する関係で 257 個以上のパケットクラスは登録できない
			throw new IllegalArgumentException("PacketPipeline: occur 256 packets has been registered");
		}
		if (this.packets.contains(clazz)) {
			System.err.println("PacketPipeline: registry same packets");
			return false;
		}
		this.packets.add(clazz);
		return true;
	}
	
	@Override
	protected void encode(ChannelHandlerContext ctx, AbstractPacket msg, List<Object> out) throws Exception {
		int discriminator = this.packets.indexOf(msg.getClass());
		if (discriminator < 0) {
			throw new NullPointerException("No Packet Registered for: " + msg.getClass().getCanonicalName());
		}
		ByteBuf buffer = Unpooled.buffer();
		buffer.writeByte((byte) discriminator); // パケットの種類を byte 値で格納
		msg.encodeInto(ctx, buffer); // パケット本文を格納
		FMLProxyPacket proxyPacket = new FMLProxyPacket(buffer.copy(), ctx.channel().attr(NetworkRegistry.FML_CHANNEL).get());
		out.add(proxyPacket);
	}
	
	@Override
	protected void decode(ChannelHandlerContext ctx, FMLProxyPacket msg, List<Object> out) throws Exception {
		ByteBuf payload = msg.payload();
		byte discriminator = payload.readByte(); // パケットの種類
		Class<? extends AbstractPacket> clazz = this.packets.get(discriminator);
		if (clazz == null) {
			throw new NullPointerException("No packet registered for discriminator: " + discriminator);
		}
		AbstractPacket pkt = clazz.newInstance();
		pkt.decodeInto(ctx, payload.slice()); // パケット本文をデコード
		
		// AbstractPacket のハンドリングメソッドを呼び出す
		switch (FMLCommonHandler.instance().getEffectiveSide()) {
			case CLIENT:
				pkt.handleClientSide(getClientPlayer());
				break;
			case SERVER:
				NetHandlerPlayServer netHandler = (NetHandlerPlayServer) ctx.channel().attr(NetworkRegistry.NET_HANDLER).get();
				pkt.handleServerSide(netHandler.playerEntity);
				break;
			default:
		}
		out.add(pkt);
	}
	
	/**
	 * パケットを登録する。
	 */
	protected abstract void initPackets();
	
	public void init(Class<?> modClass) {
		init(modClass.getName());
	}
	
	public void init(String channelName) {
		this.initializing = true;
		try {
			// チャネル登録
			this.channels = NetworkRegistry.INSTANCE.newChannel(channelName, this);
			// パケット登録
			initPackets();
			// パケット並び替え
			Collections.sort(this.packets, new Comparator<Class<? extends AbstractPacket>>() {
				// サーバとクライアントのリスト順序を揃える(IDに使う値がずれないように)
				@Override
				public int compare(Class<? extends AbstractPacket> left, Class<? extends AbstractPacket> right) {
					int com = String.CASE_INSENSITIVE_ORDER.compare(left.getCanonicalName(), right.getCanonicalName());
					if (com == 0) {
						com = left.getCanonicalName().compareTo(right.getCanonicalName());
					}
					return com;
				}
			});
		}
		finally {
			initializing = false;
		}
	}
	
	@SideOnly(Side.CLIENT)
	private EntityPlayer getClientPlayer() {
		return Minecraft.getMinecraft().thePlayer;
	}
	
	/**
	 * プレイヤー全員にpacketを送る。
	 * <p/>
	 * Adapted from CPW's code in cpw.mods.fml.common.network.simpleimpl.SimpleNetworkWrapper
	 *
	 * @param message The message to send
	 */
	public void sendToClient(AbstractPacket message) {
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGET).set(FMLOutboundHandler.OutboundTarget.ALL);
		this.channels.get(Side.SERVER).writeAndFlush(message);
	}
	
	/**
	 * あるプレイヤーにpacketを送る。
	 * <p/>
	 * Adapted from CPW's code in cpw.mods.fml.common.network.simpleimpl.SimpleNetworkWrapper
	 *
	 * @param message The message to send
	 * @param player  The player to send it to
	 */
	public void sendToClient(AbstractPacket message, EntityPlayerMP player) {
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGET).set(FMLOutboundHandler.OutboundTarget.PLAYER);
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGETARGS).set(player);
		this.channels.get(Side.SERVER).writeAndFlush(message);
	}
	
	/**
	 * ある一定範囲内にいるプレイヤーにpacketを送る。
	 * <p/>
	 * Adapted from CPW's code in cpw.mods.fml.common.network.simpleimpl.SimpleNetworkWrapper
	 *
	 * @param message The message to send
	 * @param point   The {@link cpw.mods.fml.common.network.NetworkRegistry.TargetPoint} around which to send
	 */
	public void sendToClient(AbstractPacket message, NetworkRegistry.TargetPoint point) {
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGET).set(FMLOutboundHandler.OutboundTarget.ALLAROUNDPOINT);
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGETARGS).set(point);
		this.channels.get(Side.SERVER).writeAndFlush(message);
	}
	
	/**
	 * 指定ディメンションにいるプレイヤー全員にpacketを送る。
	 * <p/>
	 * Adapted from CPW's code in cpw.mods.fml.common.network.simpleimpl.SimpleNetworkWrapper
	 *
	 * @param message     The message to send
	 * @param dimensionId The dimension id to target
	 */
	public void sendToClient(AbstractPacket message, int dimensionId) {
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGET).set(FMLOutboundHandler.OutboundTarget.DIMENSION);
		this.channels.get(Side.SERVER).attr(FMLOutboundHandler.FML_MESSAGETARGETARGS).set(dimensionId);
		this.channels.get(Side.SERVER).writeAndFlush(message);
	}
	
	/**
	 * サーバーにpacketを送る。
	 * <p/>
	 * Adapted from CPW's code in cpw.mods.fml.common.network.simpleimpl.SimpleNetworkWrapper
	 *
	 * @param message The message to send
	 */
	public void sendToServer(AbstractPacket message) {
		this.channels.get(Side.CLIENT).attr(FMLOutboundHandler.FML_MESSAGETARGET).set(FMLOutboundHandler.OutboundTarget.TOSERVER);
		this.channels.get(Side.CLIENT).writeAndFlush(message);
	}
}
